define([
    'underscore',
    'modules/new-appointment-request/resources/common/recent-visits/model',
], function(_, Model) {
    'use strict';

    /*
        {
          "hasVisitedInPastMonths" : false,
          "durationInMonths" : 12
        }
     */

    describe('Recent Visits Model', function() {
        var model;

        beforeEach(function() {
            model = new Model();
        });

        it('defaults to checking direct scheduling', function() {
            expect(model.eligibilityType).toBe('direct-eligibility');
        });

        it('can be set to request scheduling', function() {
            model.setAsRequestModel();
            expect(model.eligibilityType).toBe('request-eligibility');
        });

        it('can be set back to direct scheduling', function() {
            model.setAsRequestModel();
            model.setAsDirectModel();
            expect(model.eligibilityType).toBe('direct-eligibility');
        });

        it('generates the correct url', function() {
            var expected = 'base-url/VeteranAppointmentRequestService/v4/rest/direct-scheduling/site/123/patient/assigning-authority/testpatient/direct-eligibility/visited-in-past-months';
            var output = model.createUrl('123');
            expect(output).toBe(expected);
        });

        describe('Fetch', function() {
            beforeEach(function() {
                spyOn(model, 'fetch').and.callFake(_.noop);
            });

            it('is called with the correct parameters', function() {
                var siteCode = '123';
                var institutionCode = siteCode + 'AB';
                var clinicalService = '321';
                var args;

                model.fetchRecentVisits(siteCode, institutionCode, clinicalService);
                args = model.fetch.calls.first().args[0];

                expect(args.url).not.toBeUndefined();
                expect(args.data).not.toBeUndefined();
                expect(args.data['clinical-service']).toBe(clinicalService);
                expect(args.data['institution-code']).toBe(institutionCode);
            });
        });

        describe('hasRecentVisits', function() {
            it('returns true when there is no limit to time between visits', function() {
                var data = {durationInMonths: 0};

                model.set(data);
                expect(model.hasRecentVisit()).toBeTruthy();
            });

            it('returns true when there is a duration and the user has met the requirement', function() {
                var data = {durationInMonths: 12, hasVisitedInPastMonths: true};

                model.set(data);
                expect(model.hasRecentVisit()).toBeTruthy();
            });

            it('returns false when there is a duration and the user has not met the requirement', function() {
                var data = {durationInMonths: 12, hasVisitedInPastMonths: false};

                model.set(data);
                expect(model.hasRecentVisit()).toBeFalsy();
            });
        });


    });
});
